;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  shape-names  ;; list that holds the names of the shapes a student's turtle can have
  m  ;; the slope of the line of regression- only calculated if linear-regression? is true
  b  ;; the y-intersept of the line of regression- only calculated if linear-regression? is true
  r-square  ;; the correlation coefficient of the line of regression- only calculated if linear-regression? is true
  equation  ;; the equation for the line of regression- only calculated if linear-regression? is true
  challenge  ;; holds the current verbal challenge that the instructor gives to the users

;; quick start instructions variables
  quick-start  ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item  ;; the index of the current quickstart instruction
  qs-items  ;; the list of quickstart instructions
]

turtles-own
[
  user-id  ;; unique id generated by each calculator to identify each student or teacher turtle
]

patches-own
[
  axis?  ;; true if the pxcor or pycor are 0
  reg-point?  ;; true if the patch falls on the line of regression
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup-quick-start
  hubnet-set-client-interface "TI-83+" [ "AAA - Function Activity 2.4" [ "L1" ] ]
  hubnet-reset
  setup
end

;; Initializes the display, and creates a list that contains the names of the shapes
;; used by turtles in this activity.  The placement of the shape names in the last
;; corresponds to the numbers sent by calculators.
to setup
  clear-output
  cp
  ct
  clear-all-plots

  ;; setup and draw the axes
  ask patches
  [
    ifelse (pxcor = 0 or pycor = 0)
    [
      set pcolor 2
      set axis? true
    ]
    [
      set pcolor 0
      set axis? false
    ]
  ]

  ;; create a list of the shape names for the calculators to use
  set shape-names ["wide wedge" "square" "" "thin wedge" "" "" "" "car"]

  ;; initialize the equation for the line of regression to be nothing
  set equation ""

  ;; initially set the challenge to nothing
  set challenge ""
    
  ;; send the appropriate setup information to the calculators
  setup-hubnet
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "clear the calulator memory. (Press 2nd MEM 7 1 2)"
                 + "\n  -Open the teacher console for this activity and press the ERASE ALL DATA button."
    setup
  ]
end

;; only used in testing- creates some turtles to test the activity
to make-test-turtles [ num ]
  cct num
  [
    set heading 0
    setxy random-float screen-size-x random-float screen-size-y
    set user-id -1
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; receives information from the calculators and runs the simulation
to go
  ;; set all patches to have be not patches on the line of regression
  ask patches
  [ set reg-point? false ]

  ;; get any new information from the calculators
  listen-calc

  ;; do the calculations for determining the equation of the line of regression and correlation coefficient
  if linear-regression?
  [ lin-reg ]

  ;; remove or draw the line of regression for all patches that have reg-point? true
  ask patches
  [
    ;; draw the line of regression.  note: reg-point? can only ever be true if linear-regression? is true
    ifelse reg-point?
    [ set pcolor 13 ]
    [
      ;; draw the axes or the background to an appropriate color
      ifelse axis?
      [ set pcolor 2 ]
      [ set pcolor black ]
    ]
  ]
end

to lin-reg
  locals
  [
    ave-x  ;; the average of all the turtles' xcors
    ave-y  ;; the average of all the turtles' ycors
    sum-x  ;; the sum of all the turtles' xcors
    sum-y  ;; the sum of all the turtles' ycors
    sum-xy  ;; the sum of the product of each turtle's xcor and ycor
    sum-x-square  ;; the sum of the product of each turtle's xcor squared
    sum-y-square  ;; the sum of the product of each turtle's ycor squared
    num-turtles  ;; the number of turtles
    denominator  ;; holds the denominator for the calculations of the slope and the y-intersept
    xcors  ;; a list containing the xcors of the turtles
    ycors  ;; a list containing the ycors of the turtles
  ]

  ;; you need to have at least two points to make a line
  if count patches with [ any? turtles-here ] > 1
  [
    ;; initialize the variables
    set num-turtles count turtles
    set xcors values-from turtles [ xcor ]
    set ycors values-from turtles [ ycor ]
    set sum-x sum xcors
    set sum-y sum ycors
    set ave-x sum-x / num-turtles
    set ave-y sum-y / num-turtles
    set sum-xy sum values-from turtles [ xcor * ycor ]
    set sum-x-square sum values-from turtles [ xcor * xcor ]
    set sum-y-square sum values-from turtles [ ycor * ycor ]

    ;; compute and store the demonimator for the slope and y-intercept so we don't have to do it twice
    set denominator ( sum-x-square - num-turtles * ( ave-x ^ 2 ) )

    ;; if the denominator = 0 or the min of xcors = the max of xcors, the turtles are in a vertical line
    ;; thus, the line of regression will be undefined and have a r^2 value of 1.0
    ifelse ( denominator = 0 ) or ( min xcors = max xcors )
    [
      set m 0
      set b 0
      set equation "Undefined"
      ;; since all the turtles are in a line r^2 must be 1
      set r-square 1
      ;; set all the patches in the column to be on the line
      ask patches with [ pxcor = xcor-of one-of turtles ]
      [ set reg-point? true ]
    ]
    [
      ;; otherwise, we have some other type of line.  so find the y-intersept and slope of the line
      set b ( ( ave-y * sum-x-square ) - ( ave-x * sum-xy ) ) / denominator
      set m ( sum-xy - num-turtles * ave-x * ave-y ) / denominator

      ;; set the equation to the appropriate string
      set equation ( precision m 2 ) + "X + " + ( precision b 2 )

      ;; compute the value of r^2
      ifelse ( ( sum-y-square - num-turtles * ( ave-y ^ 2 ) ) = 0 ) or ( min ycors = max ycors )
      [ set r-square 1 ]
      [
        set r-square ( ( sum-xy - num-turtles * ave-x * ave-y ) ^ 2 ) /
          ( ( sum-x-square - num-turtles * ( ave-x ^ 2 ) ) * ( sum-y-square - num-turtles * ( ave-y ^ 2 ) ) )
      ]

      ;; set the patches' reg-point? to true if they lie on the line of regression
      ask patches with [ pycor = round ( m * pxcor + b ) ]
      [ set reg-point? true ]
    ]
  ]
end

;; sets challenge to the value that the user enters in an input dialog
to set-challenge
  set challenge user-input "Enter the Verbal Challenge:"
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; initialize instructions to quickly setup the model, calculators, and TISchool webpage to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "Optional- Zoom In (see Tools in the Menu Bar) - optional"
    "Press the GO button."
    "Everyone: Reset the RAM on your calculator (2nd MEM 7 1 2)."
    "Login to the calculator."
    "Choose the PICK POINTS option."
    "Remember your shape and color and press ENTER."
    "Move around using the arrow keys to..." "acquaint yourselves with the interface."
    "Teacher: Issue your students a challenge, such as..." "create a mock data set with an R^2 of .75."
    "Optional- Press the NetLogo SET CHALLENGE button and enter..." "the challenge to be displayed in the CURRENT CHALLENGE monitor."
    
    "Teacher: To rerun the activity, do not clear the server."
    "Stop the simulation by pressing the NetLogo GO button."
    "Press the NetLogo RE-RUN button."
    "Press the GO button."
    "Everyone: Exit to the calculator Main Menu."
    "Choose the RESTART option from the calculator Main Menu."
    "Choose the PICK POINTS option and continue."
    
    "Teacher: To start the simulation over with a new group,..." "stop the model by pressing the NetLogo GO button, if it is on."
    "Have everyone, including yourself, logout of their calculators."
    "Press the ERASE ALL DATA button on the TISCHOOL site."
    "Press the NetLogo SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start item qs-item qs-items
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start item qs-item qs-items
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start item qs-item qs-items
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the calculators ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; send the appropriate setup information to the calculators
to setup-hubnet
  locals
  [
    parameters  ;; the setup information to be sent to the calculators
  ]
  
  set parameters [ 1 ]  ;; 1 means that we can only have one point on the calculators
  set parameters lput (- screen-edge-x) parameters  ;; append the negative screen-edge-x
  set parameters lput screen-edge-x parameters  ;; append the screen-edge-x
  set parameters lput (- screen-edge-y) parameters  ;; append the negative screen-edge-y
  set parameters lput screen-edge-y parameters  ;; append the screen-edge-y

  hubnet-broadcast "PARAM" parameters
end

;; when a command is sent, this finds out which calculator sent it and then executes the command
to listen-calc
  locals
  [
    current-id
    cmd
  ]

  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    set current-id hubnet-message-source
    set cmd item 0 hubnet-message
    execute-cmd current-id cmd
  ]
end

to execute-cmd [ current-id cmd ]
  if cmd = 0
  [ exe-crt current-id ]
  if cmd = 3
  [ exe-kill current-id ]
end

; 0 Create or Move current Turtles - receives a list 6 items long - 1, shape, color, x, y, and point number (this is unused)
to exe-crt [ current-id ]
  ifelse not any? turtles with [ user-id = current-id ]
  [
    cct item 1 hubnet-message
    [
      set user-id current-id
      set heading 0
      set shape item (item 2 hubnet-message) shape-names
      set color (item 3 hubnet-message)
      setxy (item 4 hubnet-message) (item 5 hubnet-message)
    ]
  ]
  [
    ask turtles with [ user-id = current-id ]
    [ setxy (item 4 hubnet-message) (item 5 hubnet-message) ]
  ]
end

;3 Kill Turtles - kills turtle associated with id current-id
to exe-kill [ current-id ]
  ask turtles with [ user-id = current-id ]
  [ die ]
end


; ***NetLogo Model Copyright Notice***

; This activity and associated models and materials was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
; The project gratefully acknowledges the support of the 
; National Science Foundation ( REPP program ) -- grant number REC #9814682.

; Copyright 2000 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution for
; profit.

; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (2000).  NetLogo HubNet Regression model.
; http://ccl.northwestern.edu/netlogo/models/HubNetRegression.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.

; In other publications, please use:
; Copyright 1998 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetRegression
; for terms of use.
;
; ***NetLogo Model Copyright Notice***
@#$#@#$#@
GRAPHICS-WINDOW
179
148
706
520
23
15
11.0
1
12
1
1
1
0

CC-WINDOW
5
534
715
629
Command Center

BUTTON
3
10
68
43
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

BUTTON
170
60
235
93
Go
go
T
1
T
OBSERVER
T
NIL

MONITOR
2
234
69
283
Mouse X
(round mouse-xcor)
1
1

MONITOR
74
234
141
283
Mouse Y
(round mouse-ycor)
1
1

MONITOR
243
10
695
59
Quick Start Instructions (more details in info tab)
quick-start
0
1

BUTTON
461
60
539
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

BUTTON
538
60
622
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
259
60
406
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

MONITOR
2
179
177
228
Number of Points at Mouse
count turtles-at round mouse-xcor round mouse-ycor
0
1

SWITCH
3
299
175
332
linear-regression?
linear-regression?
0
1
-1000

MONITOR
2
389
52
438
R^2
r-square
2
1

MONITOR
3
337
177
386
Equation
equation
3
1

MONITOR
137
96
696
145
Current Challenge
challenge
3
1

BUTTON
4
102
129
135
Set Challenge
set-challenge
NIL
1
T
OBSERVER
T
NIL

BUTTON
76
60
157
93
Re-Run
setup
NIL
1
T
OBSERVER
T
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This model is used for an introductory approach to regressions.  Linear Regression is the fitting of a straight line through a given set of points.  Most commonly, the best-fit line is used.  The best-fit line is the line that has the highest R^2 value for this set of points.  R^2 is known as the correlation coefficient.  This value indicates how well the line of regression fits the data on a scale from 0 to 1.  The closer R^2 is to 1, the closer the line fits the data.

In this model however, you will create a set of points to create a line of regression that will have a specific R^2 value.  Issue your students a challenge, such as create a mock data set with an R^2 of .5 and then of .8.  How did the data change?  Try to data sets that have different slopes or y-intercepts.

This activity requires the TI Navigator activity AAA - Function Activity 2.4

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------

QUICKSTART INSTRUCTIONS:
------------------------
Contains instructions as to how to quickly setup the model, calculators, and TISchool web page so as to run this activity.  The instructions can be found below:

Teacher: Open the TISchool site in your web browser.
Enable activity AAA - Function Activity 2.4.
Open the teacher console and press the ERASE ALL DATA button.

If it is not open already, open the NetLogo model.
If you are prompted by a Login dialog, enter your teacher id and password and any other necessary information.
Optional- Zoom In (see Tools in the Menu Bar)
Press the GO button.

Everyone: Reset the RAM on the calculator by pressing the following keys: 2nd MEM 7 1 2.
Login to the calculator.
Choose the PICK POINTS option.
Remember your shape and color and press ENTER.
Move around using the arrow keys to acquaint yourselves with the interface.

Teacher: Issue your students a challenge, such as create a mock data set with an R^2 of .75.
Optional- Press the NetLogo SET CHALLENGE button and enter the challenge to be displayed in the CURRENT CHALLENGE monitor.

Teacher: To rerun the activity, do not clear the server.
Stop the simulation by pressing the NetLogo GO button.
Press the NetLogo RE-RUN button.
Press the GO button.
Everyone: Exit to the calculator Main Menu.
Choose the RESTART option from the calculator Main Menu.
Choose the PICK POINTS option and continue.

Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on.
Have everyone, including yourself, logout of their calculators.
Press the ERASE ALL DATA button on the TISCHOOL site.
Press the NetLogo SETUP button.
Follow these instructions from the beginning again.

BUTTONS:
--------
SETUP - clears all turtles and patches.  This should only be pressed when starting out with a new group of users since all data is lost.
RE-RUN - sets up the model to be able to run again with the same users
GO - runs the simulation
SET RULE - displays an input dialog that allows the leader to enter the verbal challenge.  The entered challenge is then displayed in the CURRENT CHALLENGE monitor.  This is meant for a convenience of the users and has no effect on the actual running of the simulation.
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

MONITORS:
---------
MOUSE X - shows the pxcor of the patch the mouse is currently on, or if the mouse is no longer over the graphics window, it displays the pxcor of the patch the mouse was last on
MOUSE Y - shows the pycor of the patch the mouse is currently on, or if the mouse is no longer over the graphics window, it displays the pycor of the patch the mouse was last on
NUMBER OF POINTS AT MOUSE - shows the number of turtles on the patch the mouse is currently on, or if the mouse is no longer over the graphics window, it displays the number of turtles on the patch the mouse was last on
EQUATION - shows the equation for the line of regression if LINEAR-REGRESSION? is true
R^2 - shows the correlation coefficient for the line of regression if LINEAR-REGRESSION? is true
CURRENT CHALLENGE- displays the last challenge that was entered when the SET CHALLENGE button was pressed.  This is meant for a convenience of the users and has no effect on the actual running of the simulation.

SWITCHES:
---------
LINEAR-REGRESSION? - if true, the line of regression is shown.  Otherwise, it is not.


CALCULATOR INFORMATION
----------------------

TEACHER CALCULATOR:
-------------------
Students and teacher have identical calculator programs.

STUDENT CALCULATORS:
--------------------
To enter the simulation, choose the PICK POINTS option.  Move your point around by pressing the ARROW keys.

The DONE softkey returns you to the Main Menu.
The STEP softkey prompts you to change the amount of distance you move when you press an ARROW key.  You can choose from 0.5, 1, and 5.


THINGS TO NOTICE
----------------
What is the least number of points to create an R^2 value 0.0?  1.0?

Why does it not make sense to have a regression line for only 1 data point?

Is it harder to get an R^2 value of a particular value when there are more or less points?  Why?

What happens to the regression line and the correlation coefficient if there are multiple points on a single patch?


THINGS TO TRY
-------------
Challenge the students to make data sets that have correlation coefficients of different values.

Move all the points the same amount in a single direction.  Does the correlation coefficient change?  The equation of the line of regression?  Why?

Find out which points have more effect on the R^2 value and the line of regression.  What qualities do these points have that other points lack?

Set LINEAR-REGRESSION? to OFF.  Challenge the students to make a data set with a certain R^2 value or a particular equation for the regression line.  Turn LINEAR-REGRESSION? ON.  How accurate were they?  What strategies did they use?  Can they be improved?


EXTENDING THE MODEL
-------------------
Support having multiple data sets.  Perhaps each calculator could control a particular data set.

Research other types of regression and support them.


CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the project:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
The project gratefully acknowledges the support of the National Science Foundation ( REPP program ) -- grant number REC #9814682.

Copyright 2000 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

Permission to use, copy, or modify this software and its associated documentation, models and curricular materials for educational and research purposes only and without fee is hereby granted, provided that this copyright notice and the original authors' names appear on all copies and supporting documentation. For any other uses of this software, in original or modified form, including, but not limited to distribution in whole or in part, specific prior permission must be obtained from Uri Wilensky and Walter Stroup. These programs shall not be used, rewritten, or adapted as the basis of a commercial or hardware product without first obtaining appropriate licenses from Uri Wilensky & Walter Stroup. We make no representations about the suitability of this software for any purpose. It is provided "as is" without express or implied warranty.

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (2000).  NetLogo HubNet Regression model. http://ccl.northwestern.edu/netlogo/models/HubNetRegression. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2000 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetRegression for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
true
0
Polygon -7566196 true true 180 15 164 21 144 39 135 60 132 74 106 87 84 97 63 115 50 141 50 165 60 225 150 285 165 285 225 285 225 15 180 15
Circle -16777216 true false 180 30 90
Circle -16777216 true false 180 180 90
Polygon -16777216 true false 80 138 78 168 135 166 135 91 105 106 96 111 89 120
Circle -7566196 true true 195 195 58
Circle -7566196 true true 195 47 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

thin wedge
true
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

wide wedge
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
